/*
Régis Lécu
20/07/2016

Création et vérification d'une signature basée sur l'algorithme RSA
Merci à Jean-Michel Doudoux pour son exemple et ses explications : 
http://www.jmdoudoux.fr/java/dej/chap-jca.htm

*/
package crypto;

import java.security.*;
import java.util.logging.Level;
import java.util.logging.Logger;


public class TestSignature 
{
    // la méthode autoriteCertification retourne une paire de clés publique/privé
    //   générée en utilisant l'algorithme "RSA"
    public static KeyPair autoriteCertification()
    {        
        KeyPair cles=null;
        try 
        {
            // crée un générateur de paires de clés publiques/privées pour RSA
            KeyPairGenerator gen = KeyPairGenerator.getInstance("RSA");
            // initialisation du générateur de paires de clés avec la taille de la clé en bits et une source d'aléa
            gen.initialize(2048, new SecureRandom());
            // création de la paire de clés
            cles = gen.generateKeyPair();                     
        }
        catch (NoSuchAlgorithmException ex) 
        {
            System.out.println(ex);
        }
        return cles;
    }

    // L'émetteur du message :
    //  construit une signature pour le message : TEXTE 
    //  avec la clée privée : CLEPRIVEE
    //  Valeur de retour : la signature
    public static byte[]  emetteur(String texte, PrivateKey clePrivee) 
    {
        byte [] tabSig = null;
        try 
        {
            // demande un outil de  signature en "SHA1 avec RSA"
            Signature signature = Signature.getInstance("SHA1withRSA");
            
            // initialise l'outil de signature avec sa clé privée, et une valeur aléatoire
            signature.initSign ( clePrivee, new SecureRandom());
                        
            // particularise l'outil de  signature pour les  données à signer
            signature.update(texte.getBytes() );
            // construit la signature   
            tabSig  = signature.sign();            
        } 
        catch (NoSuchAlgorithmException ex) 
        {  
            System.out.println(ex);
        } catch (InvalidKeyException ex) {
            System.out.println(ex);
        } catch (SignatureException ex) {
            System.out.println(ex);
        }
        return tabSig;    
    }

    public static boolean recepteur(String texte, PublicKey clePublique, byte[] sigTexte) 
    {
        boolean signatureOK = false;
        try
        {
            // demande un outil de  signature en "SHA1 avec RSA"
            Signature signature = Signature.getInstance("SHA1withRSA");
             // initialise l'outil de signature avec la clé publique (correspondant à
             // la clé privée qui a crée la signature
            signature.initVerify (clePublique);
             // particularise l'outil de  signature pour les  données à signer
            signature.update(texte.getBytes() );
            // vérifie la signature envoyée par l'émetteur
            signatureOK = signature.verify(sigTexte);            
        } 
        catch (NoSuchAlgorithmException ex)
        {
            System.out.println(ex);
        } 
        catch (InvalidKeyException ex)
        {
            System.out.println(ex);
        } 
        catch (SignatureException ex) 
        {
             System.out.println(ex);
        }
        return signatureOK;
    }

    public static void main(String[] args)
    {
        System.out.println("=== Envoi d'un message signé et reception avec vérification de la signature ===");
        try 
        {
            String message = "Bonjour Régis";
            KeyPair cles = autoriteCertification();
           
            
            // l'émetteur du message construit sa signature SIGMES :
            //  à partir du message et de sa clé privée.
            byte [] sigmes = emetteur(message, cles.getPrivate());
            
            // Le récepteur vérifie l'identité de l'émetteur du message en
            // avec la signature du message et la clé publique de l'émetteur
            // Premier test OK : bon message, la bonne clé, la bonne signature
            System.out.println("Test 1 = > Validation : " + recepteur(message, cles.getPublic(), sigmes) );
            
            // Test 2 : faux. Le message a été modifié : pas d'accent sur Régis !
            System.out.println("Test 2 = > Validation : " + recepteur("Bonjour Regis", cles.getPublic(), sigmes) );
            
            // Test 3 : faux. Ce n'est pas la clé publique qui correspond à la clé privée de l'émetteur
            //  RSA détecte l'erreur
            KeyPair badcles = autoriteCertification();
            System.out.println("Test 3 = > Validation : " + recepteur("Bonjour Régis", badcles.getPublic(), sigmes) );
            
            // Test 4 : faux. C'est le bon message, et la bonne clé publique mais la signature du message a été corrompue            
            sigmes[0]++;
            System.out.println("Test 4 = > Validation : " + recepteur("Bonjour Régis", cles.getPublic(), sigmes) );      
          
        }
        catch (Exception ex) 
        {
             System.out.println(ex);
        }
    }

}
